/**
 * \file keyman_arch.h
 *
 * \brief Definition of the functions to be realized by each architecture
 *
 * \author Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2015 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

#ifndef __LIBKEYMAN_ARCH_H_
#define __LIBKEYMAN_ARCH_H_

#include <private/sdc_intern.h>

/**
 * \defgroup libKeyman_Arch_API Architecture specific API of Libkeyman
 * Documentation of functions which need to be implemented for each architecture.
 *
 * Please note: Only privileged applications (i.e. SDC daemon) must be able to perform certain operations like
 *  - \ref keyman_empty_persistent_keystore
 *  - \ref keyman_flush_keystore
 *  - \ref keyman_lock_unlock_keystore
 *  - \ref keyman_generate_plain_storage_key_explicit_kid
 *  - \ref keyman_generate_plain_storage_key_automatic_kid
 *  - \ref keyman_generate_random_storage_key_explicit_kid
 *  - \ref keyman_generate_random_storage_key_automatic_kid
 *  - \ref keyman_import_insecure_product_key
 *  - \ref keyman_remove_storage_key
 * Failing to do so could result e.g. in an application removing the key bypassing all permission checks.
 * One way to enforce this is using credentials of the corresponding kernel interface.
 * As the SDC daemon is the only intended user it is OK to enforce this for all functions.
 */

/**
 * \ingroup libKeyman_Arch_API
 * \brief Used to specify that this function is part of the architecture specific API of libKeyman
 *
 * These function will be visible to applications including libkeyman library */
#define KEYMAN_ARCH_API __attribute__((__visibility__("internal")))

/**
 * \ingroup libKeyman_Arch_API
 * \brief Verify the kernel version of the libKeyman kernel interface.
 *
 * No initialization of the libkeyman context is required
 *
 * \param[out] version_info_str optional pointer to return detailed error information (externally freed)
 *
 * \return SDC_OK on success
 * \return SDC_NOT_SUPPORTED if function is not implemented
 * \return else appropriate error value
 */
KEYMAN_ARCH_API sdc_error_t keyman_arch_kernel_version_verify(char **version_info_str);

/**
 * \ingroup libKeyman_Arch_API
 * \struct libkeyman_ctx
 * \brief Architecture specific struct to collect all information of the context
 *
 * Examples:
 *  - File descriptor of the opened kernel interface
 */

/**
 * \ingroup libKeyman_Arch_API
 * \brief allocate a architecture specific context
 *
 * This function can be used to allocate an initialize the context before any operation is done.
 * E.g. Within this function the kernel interface could be opened and the opened file descriptor
 * could be stored within \ref libkeyman_ctx.
 *
 * The architecture independent wrapper will assure that \p ctx != NULL.
 *
 * Please refer to \ref keyman_ctx_alloc for details on the parameters and return values.
 */
KEYMAN_ARCH_API sdc_error_t keyman_arch_ctx_alloc(libkeyman_ctx_t **ctx);

/**
 * \ingroup libKeyman_Arch_API
 * \brief free the architecture specific context allocated using \ref keyman_ctx_alloc
 *
 * This function can be used to deallocate the context after all operation is done.
 * E.g. Within this function the kernel interface opened in \ref keyman_arch_ctx_alloc could be closed
 *
 * The architecture independent wrapper will assure that \p ctx != NULL.
 *
 * Please refer to \ref keyman_ctx_free for details on the parameters and return values.
 */
KEYMAN_ARCH_API sdc_error_t keyman_arch_ctx_free(libkeyman_ctx_t *ctx);

/**
 * \ingroup libKeyman_Arch_API
 * \brief Restore the architecture specific keystore
 *
 * Note: The format of the keystore can be architecture specific.
 * The function needs to guarantee that unprivileged callers can't modify the content of the keystore.
 *
 * The architecture independent wrapper will assure that \p ctx != NULL and \p fd >= 0
 *
 * Please refer to \ref keyman_load_persistent_keystore for details on the parameters and return values.
 */
KEYMAN_ARCH_API sdc_error_t keyman_arch_load_persistent_keystore(libkeyman_ctx_t *ctx, int fd,
                                                                 size_t *loaded_keys, size_t *existing_keys);

/**
 * \ingroup libKeyman_Arch_API
 * \brief Store the architecture specific keystore
 *
 * Note: The format of the keystore can be architecture specific.
 * The function needs to guarantee that unpriviledged callers can't modify the content of the keystore.
 * The generated format must be restorable using \ref keyman_arch_load_persistent_keystore.
 *
 * The architecture independent wrapper will assure that \p ctx != NULL and \p fd >= 0
 *
 * Please refer to \ref keyman_store_persistent_keystore for details on the parameters and return values.
 */
KEYMAN_ARCH_API sdc_error_t keyman_arch_store_persistent_keystore(libkeyman_ctx_t *ctx, int fd,
                                                                  size_t *stored_keys);

/**
 * \ingroup libKeyman_Arch_API
 * \brief Create an empty keystore
 *
 * Note: The format of the keystore can be architecture specific.
 * The function needs to guarantee that unprivileged callers can't trigger this action.
 * The generated format must be restorable using \ref keyman_arch_load_persistent_keystore.
 *
 * The architecture independent wrapper will assure that \p ctx != NULL and \p fd >= 0
 *
 * Please refer to \ref keyman_empty_persistent_keystore for details on the parameters and return values.
 */
KEYMAN_ARCH_API sdc_error_t keyman_arch_empty_persistent_keystore(libkeyman_ctx_t *ctx, int fd);

/**
 * \ingroup libKeyman_Arch_API
 * \brief Flush the keystore in memory
 *
 * Note: The function needs to guarantee that unprivileged callers can't trigger this action.
 *
 * The architecture independent wrapper will assure that \p ctx != NULL and \p flush_content is valid
 *
 * Please refer to \ref keyman_flush_keystore for details on the parameters and return values.
 */
KEYMAN_ARCH_API sdc_error_t keyman_arch_flush_keystore(libkeyman_ctx_t *ctx,
                                                       libkeyman_keystore_cont_t flush_content, bool force);

/**
 * \ingroup libKeyman_Arch_API
 * \brief Lock unlock the keystore in memory
 *
 * Note: The function needs to guarantee that unprivileged callers can't trigger this action.
 * Note: While the keystore is locked no application needs to be able to reference/assign any locked key
 *
 * The architecture independent wrapper will assure that \p ctx != NULL and \p flush_content is valid
 *
 * Please refer to \ref keyman_flush_keystore for details on the parameters and return values.
 */
KEYMAN_ARCH_API sdc_error_t keyman_arch_lock_unlock_keystore(libkeyman_ctx_t *ctx,
                                                             libkeyman_keystore_cont_t keystore_content, bool lock);

/**
 * \ingroup libKeyman_Arch_API
 * \brief Install a key with an explicit/automatic KID and a known/random secret
 *
 * Note: The function needs to guarantee that unprivileged callers can't trigger this action.
 * Note: While a key is not activated no application needs to be able to reference/assign the key.
 * Note: Lock state of the keystore is not relevant for key installation.
 *
 * The architecture independent wrapper will assure that
 *  - \p ctx != NULL
 *  - \p key_id->first != \ref SDC_FLAG_INVALID_KID
 *  - \p key_id->last != \ref SDC_FLAG_INVALID_KID, key_id_last==key_id_first for explicit KID
 *  - \p key_id->result != NULL for automatic KID, else NULL
 *  - \p key_spec != NULL
 *  - \p key_spec->fmt valid member of \ref sdc_key_fmt_t
 *  - \p key_spec->len valid member of \ref sdc_key_len_t
 *  - \p key_spec->permissions != NULL
 *  - \p key_spec->permissions is in a valid range
 *  - \p key_secret->enc is a valid member of \ref sdc_keysecret_enc_t
 *  - \p key_secret->secret != NULL for known secret, else random secret
 *  - \p key_secret->secret_len > 0
 *
 * Please refer to:
 *  - \ref keyman_generate_plain_storage_key_explicit_kid
 *  - \ref keyman_generate_plain_storage_key_automatic_kid
 *  - \ref keyman_generate_random_storage_key_explicit_kid
 *  - \ref keyman_generate_random_storage_key_automatic_kid
 * for details on the parameters and return values.
 */
KEYMAN_ARCH_API sdc_error_t keyman_arch_generate_storage_key(
    libkeyman_ctx_t *ctx,
    const sdc_key_id_range_t *key_id,
    const libkeyman_key_spec_t *key_spec,
    bool persistent,
    bool activate_immediately,
    const sdc_keysecret_t *key_secret);

/**
 * \ingroup libKeyman_Arch_API
 * \brief Import a wrapped key with automatically determined KID into the keystore
 *
 * Note: The function needs to guarantee that unprivileged callers can't trigger this action.
 * Note: While a key is not activated no application needs to be able to reference/assign the key.
 * Note: Lock state of the keystore is not relevant for key installation.
 *
 * The functions \ref keyman_intern_search_kid_init and \ref keyman_intern_search_kid_next can
 * be used to get a random initial value and iterate through the range of \p key_id_first to key_id_last
 * when trying to find a free key identifier
 *
 * The architecture independent wrapper will assure that
 *  - \p ctx != NULL
 *  - \p key_id->first != \ref SDC_FLAG_INVALID_KID
 *  - \p key_id->last != \ref SDC_FLAG_INVALID_KID, key_id_last==key_id_first for explicit KID
 *  - \p key_id->result != NULL for automatic KID, else NULL
 *  - \p key_spec != NULL
 *  - \p key_spec->fmt valid member of \ref sdc_key_fmt_t
 *  - \p key_spec->len valid member of \ref sdc_key_len_t
 *  - \p key_spec->permissions != NULL
 *  - \p key_spec->permissions is in a valid range
 *  - \p key_secret->enc is a valid member of \ref sdc_keysecret_enc_t
 *  - \p key_secret->secret != NULL
 *  - \p key_secret->secret_len > 0
 *
 * Please refer to \ref keyman_import_storage_key_automatic_kid for details on the parameters and return values.
 */
KEYMAN_ARCH_API sdc_error_t keyman_arch_import_storage_key(
    libkeyman_ctx_t *ctx,
    const sdc_key_id_range_t *key_id,
    const libkeyman_key_spec_t *key_spec,
    bool persistent,
    bool activate_immediately,
    const sdc_keysecret_t *key_secret);

/**
 * \ingroup libKeyman_Arch_API
 * \brief Special function to import an product key (insecure way - plain secret)
 *
 * This function is similar to \ref keyman_generate_plain_storage_key_explicit_kid.
 * In contrast to \ref keyman_generate_plain_storage_key_explicit_kid it must be called
 * during the initial boot of the target (at the end of production) only.
 * It will be used to install common keys to all the devices while being in a secure environment.
 * Furthermore the keys should be marked as product keys explicitly.
 * Note: In extremes the target will reboot during the initial boot. So this function might
 * be called in multiple power cycles. Anyway after the import of all keys has finished once,
 * it must not be called again.
 *
 * Note: The function needs to guarantee that unprivileged callers can't trigger this action.
 * Note: While a key is not activated no application needs to be able to reference/assign the key.
 * Note: Lock state of the keystore is not relevant for key import.
 *
 * The architecture independent wrapper will assure that
 *  - \p ctx != NULL
 *  - \p key_id != \ref SDC_FLAG_INVALID_KID
 *  - \p key_spec != NULL
 *  - \p key_spec->fmt valid member of \ref sdc_key_fmt_t
 *  - \p key_spec->len valid member of \ref sdc_key_len_t
 *  - \p key_spec->permissions != NULL
 *  - \p key_spec->permissions is in a valid range
 *  - \p key_secret->enc is a valid member of \ref sdc_keysecret_enc_t
 *  - \p key_secret->secret != NULL
 *  - \p key_secret->secret_len > 0
 *
 * Please refer to \ref keyman_import_insecure_product_key for details on the parameters and return values.
 */
KEYMAN_ARCH_API sdc_error_t keyman_arch_import_insecure_product_key(
    libkeyman_ctx_t *ctx,
    sdc_key_id_t key_id,
    const libkeyman_key_spec_t *key_spec,
    bool activate_immediately,
    const sdc_keysecret_t *key_secret);

/**
 * \ingroup libKeyman_Arch_API
 * \brief Activate key
 *
 * Note: The function needs to guarantee that unprivileged callers can't trigger this action.
 * Note: While a key is not activated no application needs to be able to reference/assign the key.
 *
 * The architecture independent wrapper will assure that
 *  - \p ctx != NULL
 *  - \p key_id != \ref SDC_FLAG_INVALID_KID
 *
 * Please refer to \ref keyman_activate_key for details on the parameters and return values.
 */
KEYMAN_ARCH_API sdc_error_t keyman_arch_activate_key(
    libkeyman_ctx_t *ctx,
    sdc_key_id_t key_id);

/**
 * \ingroup libKeyman_Arch_API
 * \brief Remove a key-storage key
 *
 * This function is similar to \ref keyman_generate_plain_storage_key_explicit_kid.
 * In contrast to \ref keyman_generate_plain_storage_key_explicit_kid it must be called
 * during the initial boot of the target (at the end of production) only.
 * It will be used to install common keys to all the devices while being in a secure environment.
 * Furthermore the keys should be marked as product keys explicitly.
 * Note: In extremes the target will reboot during the initial boot. So this function might
 * be called in multiple power cycles. Anyway after the import of all keys has finished once,
 * it must not be called again.
 *
 * Note: The function needs to guarantee that unprivileged callers can't trigger this action.
 * Note: A key can't be deleted in case it is assigned to any session in libSDC
 * Note: Lock state of the keystore and activate state of the key are not relevant for key removal.
 *
 * The architecture independent wrapper will assure that
 *  - \p ctx != NULL
 *  - \p result_is_persistent != NULL
 *  - \p key_id != \ref SDC_FLAG_INVALID_KID
 *
 * Please refer to \ref keyman_remove_storage_key for details on the parameters and return values.
 */
KEYMAN_ARCH_API sdc_error_t keyman_arch_remove_storage_key(
    libkeyman_ctx_t *ctx,
    sdc_key_id_t key_id,
    uid_t uid, gid_t gid,
    bool *result_is_persistent
    );

#endif
